import { createTRPCRouter, protectedProcedure } from "@/backend/trpc/create-context";
import { z } from "zod";
import OpenAI from 'openai';
import { getApiKeys } from "@/backend/db/admin";

export const aiRouter = createTRPCRouter({
  /**
   * Generate story content using OpenAI GPT-4
   */
  generateStoryContent: protectedProcedure
    .input(z.object({
      childName: z.string(),
      situation: z.string(),
      complexity: z.enum(["very-simple", "simple", "moderate"]),
      tone: z.enum(["friendly", "calm", "encouraging", "straightforward"]),
    }))
    .mutation(async ({ input }) => {
      const apiKeys = await getApiKeys();

      if (!apiKeys.openaiKey) {
        throw new Error('OpenAI API key not configured. Please configure it in the admin panel.');
      }

      const openai = new OpenAI({ apiKey: apiKeys.openaiKey });

      // Build the prompt based on input parameters
      const complexityInstructions = {
        "very-simple": "Use very simple words and short sentences (3-5 words each). Perfect for young children or those with learning difficulties.",
        "simple": "Use simple language and short sentences (5-8 words each). Easy to understand.",
        "moderate": "Use clear language with moderate sentence length (8-12 words). Include some descriptive details."
      };

      const toneInstructions = {
        "friendly": "Use a warm, friendly, and reassuring tone.",
        "calm": "Use a gentle, peaceful, and calming tone.",
        "encouraging": "Use a positive, motivating, and supportive tone.",
        "straightforward": "Use a clear, direct, and matter-of-fact tone."
      };

      const prompt = `Create a social story for a child named ${input.childName} about ${input.situation}.

Guidelines:
- ${complexityInstructions[input.complexity]}
- ${toneInstructions[input.tone]}
- Use "I" statements from the child's perspective
- Include what will happen, how the child might feel, and what they can do
- Make it practical and actionable
- Keep the total story to 3-5 short paragraphs
- Focus on helping the child understand and prepare for this situation

Write the story now:`;

      try {
        const completion = await openai.chat.completions.create({
          model: 'gpt-4',
          messages: [
            {
              role: 'system',
              content: 'You are a helpful assistant that creates social stories for children. Social stories help children understand social situations and what to expect. They should be clear, supportive, and age-appropriate.'
            },
            {
              role: 'user',
              content: prompt
            }
          ],
          temperature: 0.7,
          max_tokens: 500,
        });

        const content = completion.choices[0]?.message?.content;

        if (!content) {
          throw new Error('No content generated from OpenAI');
        }

        return {
          content,
          usage: {
            promptTokens: completion.usage?.prompt_tokens || 0,
            completionTokens: completion.usage?.completion_tokens || 0,
            totalTokens: completion.usage?.total_tokens || 0,
          }
        };
      } catch (error: any) {
        console.error('OpenAI API error:', error);
        throw new Error(`Failed to generate story: ${error.message || 'Unknown error'}`);
      }
    }),

  /**
   * Generate image using OpenAI DALL-E 3
   */
  generateImage: protectedProcedure
    .input(z.object({
      prompt: z.string(),
      style: z.enum(["cartoon", "realistic", "minimal", "illustrated"]).optional(),
    }))
    .mutation(async ({ input }) => {
      const apiKeys = await getApiKeys();

      if (!apiKeys.openaiKey) {
        throw new Error('OpenAI API key not configured. Please configure it in the admin panel.');
      }

      const openai = new OpenAI({ apiKey: apiKeys.openaiKey });

      // Enhance prompt based on style preference
      const styleDescriptions = {
        "cartoon": "in a friendly cartoon style, colorful and child-appropriate",
        "realistic": "in a realistic but gentle photographic style, child-friendly",
        "minimal": "in a simple, minimal illustration style with clean lines",
        "illustrated": "in a detailed illustrated storybook style"
      };

      const enhancedPrompt = input.style
        ? `${input.prompt}, ${styleDescriptions[input.style]}`
        : input.prompt;

      try {
        const response = await openai.images.generate({
          model: 'dall-e-3',
          prompt: enhancedPrompt,
          size: '1024x1024',
          quality: 'standard',
          n: 1,
          response_format: 'b64_json',
        });

        const imageData = response.data[0];

        if (!imageData.b64_json) {
          throw new Error('No image data received from OpenAI');
        }

        return {
          image: {
            base64Data: imageData.b64_json,
            mimeType: 'image/png',
          },
          revisedPrompt: imageData.revised_prompt,
        };
      } catch (error: any) {
        console.error('DALL-E API error:', error);

        // Provide helpful error messages
        if (error.message?.includes('billing')) {
          throw new Error('OpenAI API billing issue. Please check your OpenAI account billing settings.');
        } else if (error.message?.includes('rate limit')) {
          throw new Error('Rate limit exceeded. Please try again in a moment.');
        } else if (error.message?.includes('content policy')) {
          throw new Error('Image prompt was rejected by content policy. Please try a different description.');
        }

        throw new Error(`Failed to generate image: ${error.message || 'Unknown error'}`);
      }
    }),

  /**
   * Generate multiple image prompts for a story
   */
  generateImagePrompts: protectedProcedure
    .input(z.object({
      storyContent: z.string(),
      childName: z.string(),
      situation: z.string(),
      imageCount: z.number().min(1).max(5).default(3),
    }))
    .mutation(async ({ input }) => {
      const apiKeys = await getApiKeys();

      if (!apiKeys.openaiKey) {
        throw new Error('OpenAI API key not configured. Please configure it in the admin panel.');
      }

      const openai = new OpenAI({ apiKey: apiKeys.openaiKey });

      const prompt = `Based on this social story about ${input.childName} and ${input.situation}, generate ${input.imageCount} image prompts that would illustrate key moments or concepts in the story.

Story:
${input.storyContent}

Generate ${input.imageCount} short, descriptive image prompts (one per line) that:
- Are child-appropriate and positive
- Show the situation or activity described
- Feature diverse, inclusive characters
- Are safe for DALL-E content policy
- Each prompt should be 15-30 words

Format: Return ONLY the prompts, one per line, numbered.`;

      try {
        const completion = await openai.chat.completions.create({
          model: 'gpt-4',
          messages: [
            {
              role: 'system',
              content: 'You are a helpful assistant that creates image prompts for children\'s social stories. Generate clear, positive, child-appropriate prompts.'
            },
            {
              role: 'user',
              content: prompt
            }
          ],
          temperature: 0.8,
          max_tokens: 300,
        });

        const content = completion.choices[0]?.message?.content;

        if (!content) {
          throw new Error('No prompts generated from OpenAI');
        }

        // Parse the numbered prompts
        const prompts = content
          .split('\n')
          .filter(line => line.trim().length > 0)
          .map(line => line.replace(/^\d+\.\s*/, '').trim())
          .filter(prompt => prompt.length > 0)
          .slice(0, input.imageCount);

        return { prompts };
      } catch (error: any) {
        console.error('OpenAI API error:', error);
        throw new Error(`Failed to generate image prompts: ${error.message || 'Unknown error'}`);
      }
    }),
});
