import { trpcServer } from "@hono/trpc-server";
import { Hono } from "hono";
import { cors } from "hono/cors";
import { secureHeaders } from "hono/secure-headers";
import { mkdirSync, existsSync } from 'fs';
import { join } from 'path';

import { appRouter } from "./trpc/app-router";
import { createContext } from "./trpc/create-context";
import { getDatabase } from "./db/connection";
import { initializeDefaultAdmin } from "./db/users";
import { initializeDefaultContent } from "./db/content";
import { printEnvironmentStatus } from "./utils/env-validation";
import { logger } from "./utils/logger";

// Track server start time for uptime reporting
const serverStartTime = Date.now();

if (typeof window === 'undefined') {
  try {
    printEnvironmentStatus();

    const dbDir = join(process.cwd(), 'database');
    if (!existsSync(dbDir)) {
      logger.info('Creating database directory...', { path: dbDir });
      mkdirSync(dbDir, { recursive: true });
    }

    logger.info('Initializing database...');
    getDatabase();
    logger.info('Database initialized successfully');

    initializeDefaultAdmin();
    initializeDefaultContent();

    logger.info('Backend initialization complete');
  } catch (err) {
    logger.error('Failed to initialize backend', err);
  }
}

const app = new Hono();

// Validate CORS origins before starting
const allowedOrigins = process.env.ALLOWED_ORIGINS?.split(',').map(o => o.trim()).filter(o => o.length > 0);

if (!allowedOrigins || allowedOrigins.length === 0) {
  throw new Error('CRITICAL: ALLOWED_ORIGINS environment variable is required. Example: https://yourdomain.com,https://www.yourdomain.com');
}

// Get Supabase URL for CSP
const supabaseUrl = process.env.EXPO_PUBLIC_SUPABASE_URL || 'https:';

app.use("*", secureHeaders({
  contentSecurityPolicy: {
    defaultSrc: ["'self'"],
    scriptSrc: ["'self'"],  // Removed unsafe-inline and unsafe-eval for security
    styleSrc: ["'self'", "'unsafe-inline'"],  // Keep unsafe-inline for styles only (React Native Web needs it)
    imgSrc: ["'self'", "data:", "https:", "blob:"],
    connectSrc: ["'self'", supabaseUrl, "https://api.openai.com"],
    fontSrc: ["'self'", "data:"],
    objectSrc: ["'none'"],
    baseUri: ["'self'"],
    formAction: ["'self'"],
    frameAncestors: ["'none'"],
  },
  xFrameOptions: "DENY",
  xContentTypeOptions: "nosniff",
  referrerPolicy: "strict-origin-when-cross-origin",
  strictTransportSecurity: "max-age=31536000; includeSubDomains",
}));

app.use("*", cors({
  origin: allowedOrigins,
  credentials: true,
  allowMethods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowHeaders: ['Content-Type', 'Authorization'],
  exposeHeaders: ['Content-Length'],
  maxAge: 86400,
}));

// Request logging middleware
app.use("*", async (c, next) => {
  const start = Date.now();
  const method = c.req.method;
  const path = c.req.path;

  try {
    await next();
  } finally {
    const duration = Date.now() - start;
    const status = c.res.status;

    // Skip logging for health checks to reduce noise
    if (path !== '/health') {
      logger.request(method, path, status, duration, {
        userAgent: c.req.header('user-agent'),
      });
    }
  }
});

// Global error handler
app.onError((err, c) => {
  const status = err instanceof Error && 'status' in err ? (err as any).status : 500;

  logger.error('Unhandled error', err, {
    method: c.req.method,
    path: c.req.path,
    status,
  });

  // Don't expose internal error details in production
  const message = process.env.NODE_ENV === 'production'
    ? 'Internal server error'
    : err.message;

  return c.json({
    error: message,
    status,
    ...(process.env.NODE_ENV !== 'production' && { stack: err.stack }),
  }, status);
});

// 404 handler
app.notFound((c) => {
  logger.warn('Route not found', {
    method: c.req.method,
    path: c.req.path,
  });

  return c.json({
    error: 'Not found',
    status: 404,
    path: c.req.path,
  }, 404);
});

app.use(
  "/api/trpc/*",
  trpcServer({
    endpoint: "/api/trpc",
    router: appRouter,
    createContext,
  }),
);

app.get("/", (c) => {
  return c.json({ status: "ok", message: "SocialStoryAI API is running" });
});

app.get("/health", (c) => {
  const checks: Record<string, { status: string; message?: string }> = {};
  let isHealthy = true;

  // Check database connection
  try {
    getDatabase();
    checks.database = { status: 'ok' };
  } catch (error) {
    checks.database = { status: 'error', message: String(error) };
    isHealthy = false;
  }

  // Check required environment variables
  const requiredVars = ['JWT_SECRET', 'SUPABASE_SERVICE_ROLE_KEY', 'ENCRYPTION_KEY'];
  const missingVars = requiredVars.filter(v => !process.env[v]);

  if (missingVars.length > 0) {
    checks.environment = { status: 'error', message: `Missing: ${missingVars.join(', ')}` };
    isHealthy = false;
  } else {
    checks.environment = { status: 'ok' };
  }

  // Calculate uptime
  const uptimeMs = Date.now() - serverStartTime;
  const uptimeSeconds = Math.floor(uptimeMs / 1000);
  const uptimeMinutes = Math.floor(uptimeSeconds / 60);
  const uptimeHours = Math.floor(uptimeMinutes / 60);
  const uptimeDays = Math.floor(uptimeHours / 24);

  const uptimeStr = uptimeDays > 0
    ? `${uptimeDays}d ${uptimeHours % 24}h ${uptimeMinutes % 60}m`
    : uptimeHours > 0
      ? `${uptimeHours}h ${uptimeMinutes % 60}m ${uptimeSeconds % 60}s`
      : `${uptimeMinutes}m ${uptimeSeconds % 60}s`;

  const response = {
    status: isHealthy ? 'healthy' : 'unhealthy',
    timestamp: new Date().toISOString(),
    version: process.env.npm_package_version || '1.0.0',
    environment: process.env.NODE_ENV || 'development',
    uptime: uptimeStr,
    uptimeMs,
    checks,
  };

  return c.json(response, isHealthy ? 200 : 503);
});

// Readiness probe (for orchestration systems)
app.get("/ready", (c) => {
  try {
    getDatabase();
    return c.json({ ready: true });
  } catch {
    return c.json({ ready: false }, 503);
  }
});

export default app;
