import { Platform } from 'react-native';

/**
 * Accessibility utilities for web keyboard navigation and screen readers
 */

export interface A11yProps {
  accessibilityLabel?: string;
  accessibilityHint?: string;
  accessibilityRole?: string;
  accessible?: boolean;
}

export interface KeyboardProps {
  onKeyPress?: (event: any) => void;
  tabIndex?: number;
  role?: string;
  'aria-label'?: string;
  'aria-describedby'?: string;
}

/**
 * Generate accessibility props for a component
 */
export function a11yProps(label: string, hint?: string, role?: string): A11yProps {
  return {
    accessibilityLabel: label,
    accessibilityHint: hint,
    accessibilityRole: role as any,
    accessible: true,
  };
}

/**
 * Add keyboard navigation support for web (Enter/Space to activate)
 */
export function keyboardProps(
  onPress: () => void,
  label?: string,
  role: string = 'button'
): KeyboardProps {
  if (Platform.OS !== 'web') {
    return {};
  }

  return {
    tabIndex: 0,
    role,
    'aria-label': label,
    onKeyPress: (event: any) => {
      if (event.key === 'Enter' || event.key === ' ') {
        event.preventDefault();
        onPress();
      }
    },
  };
}

/**
 * Combine a11y and keyboard props for interactive elements
 */
export function interactiveProps(
  label: string,
  onPress: () => void,
  options?: {
    hint?: string;
    role?: string;
    disabled?: boolean;
  }
): A11yProps & KeyboardProps {
  const role = options?.role || 'button';

  return {
    ...a11yProps(label, options?.hint, role),
    ...keyboardProps(onPress, label, role),
    ...(options?.disabled && {
      accessibilityState: { disabled: true },
      'aria-disabled': true,
      tabIndex: -1,
    }),
  };
}

/**
 * Props for link elements
 */
export function linkProps(label: string, url: string): A11yProps & KeyboardProps {
  return {
    ...a11yProps(label, `Navigate to ${url}`, 'link'),
    ...(Platform.OS === 'web' && {
      tabIndex: 0,
      role: 'link',
      'aria-label': label,
    }),
  };
}

/**
 * Props for text input fields
 */
export function inputProps(
  label: string,
  options?: {
    placeholder?: string;
    required?: boolean;
    invalid?: boolean;
    errorMessage?: string;
  }
): A11yProps & Record<string, any> {
  const props: any = {
    ...a11yProps(label, options?.placeholder, 'text'),
  };

  if (Platform.OS === 'web') {
    props['aria-label'] = label;
    props['aria-required'] = options?.required || false;

    if (options?.invalid && options?.errorMessage) {
      props['aria-invalid'] = true;
      props['aria-describedby'] = `${label}-error`;
    }
  }

  return props;
}

/**
 * Props for headings (h1, h2, h3, etc.)
 */
export function headingProps(level: 1 | 2 | 3 | 4 | 5 | 6, text: string): A11yProps & Record<string, any> {
  const props: any = {
    ...a11yProps(text, undefined, 'header'),
  };

  if (Platform.OS === 'web') {
    props.role = 'heading';
    props['aria-level'] = level;
  }

  return props;
}

/**
 * Skip navigation link props for accessibility
 */
export function skipNavProps(targetId: string): KeyboardProps {
  if (Platform.OS !== 'web') {
    return {};
  }

  return {
    tabIndex: 0,
    role: 'link',
    'aria-label': 'Skip to main content',
    onKeyPress: (event: any) => {
      if (event.key === 'Enter') {
        event.preventDefault();
        const target = document.getElementById(targetId);
        if (target) {
          target.focus();
          target.scrollIntoView();
        }
      }
    },
  };
}

/**
 * Announce message to screen readers
 */
export function announce(message: string, priority: 'polite' | 'assertive' = 'polite'): void {
  if (Platform.OS !== 'web') {
    return;
  }

  // Create a live region if it doesn't exist
  let liveRegion = document.getElementById('a11y-announcer');

  if (!liveRegion) {
    liveRegion = document.createElement('div');
    liveRegion.id = 'a11y-announcer';
    liveRegion.setAttribute('role', 'status');
    liveRegion.setAttribute('aria-live', priority);
    liveRegion.setAttribute('aria-atomic', 'true');
    liveRegion.style.position = 'absolute';
    liveRegion.style.left = '-10000px';
    liveRegion.style.width = '1px';
    liveRegion.style.height = '1px';
    liveRegion.style.overflow = 'hidden';
    document.body.appendChild(liveRegion);
  }

  // Clear previous content and announce new message
  liveRegion.textContent = '';
  setTimeout(() => {
    if (liveRegion) {
      liveRegion.textContent = message;
    }
  }, 100);
}

/**
 * Focus management utility
 */
export function focusElement(elementId: string): void {
  if (Platform.OS !== 'web') {
    return;
  }

  setTimeout(() => {
    const element = document.getElementById(elementId);
    if (element) {
      element.focus();
    }
  }, 100);
}

/**
 * Props for modal/dialog components
 */
export function modalProps(
  title: string,
  description?: string,
  isOpen?: boolean
): A11yProps & Record<string, any> {
  const props: any = {
    ...a11yProps(title, description, 'alert'),
  };

  if (Platform.OS === 'web') {
    props.role = 'dialog';
    props['aria-modal'] = true;
    props['aria-label'] = title;

    if (description) {
      props['aria-describedby'] = `${title}-description`;
    }
  }

  return props;
}
