import { httpLink } from "@trpc/client";
import { createTRPCReact } from "@trpc/react-query";
import superjson from "superjson";

import type { AppRouter } from "@/backend/trpc/app-router";

export const trpc = createTRPCReact<AppRouter>();

let authToken = "";

export const setAuthToken = (token: string) => {
  authToken = token;
};

export const getAuthToken = () => authToken;

const getBaseUrl = () => {
  // For web, prefer window.location.origin for same-domain deployment
  if (typeof window !== 'undefined') {
    return process.env.EXPO_PUBLIC_API_URL || window.location.origin;
  }

  // For native apps, require explicit API URL configuration
  const url = process.env.EXPO_PUBLIC_API_URL;
  if (!url) {
    // Fallback to localhost for development
    return 'http://localhost:3000';
  }

  return url;
};

export const trpcClient = trpc.createClient({
  links: [
    httpLink({
      url: `${getBaseUrl()}/api/trpc`,
      transformer: superjson,
      headers() {
        return {
          authorization: authToken ? `Bearer ${authToken}` : "",
        };
      },
      fetch(url, options) {
        return fetch(url, options).then(res => {
          if (!res.ok) {
            return res.text().then(text => {
              console.error('TRPC Error:', res.status, text);
              throw new Error(`HTTP ${res.status}: ${text}`);
            });
          }
          return res;
        }).catch(err => {
          console.error('TRPC Fetch Error:', err);
          throw err;
        });
      },
    }),
  ],
});
