/**
 * Structured Logger for SocialStoryAI
 *
 * Provides consistent, production-ready logging with:
 * - Log levels (debug, info, warn, error)
 * - Structured JSON output for production
 * - Timestamps and context
 * - Environment-based filtering
 */

type LogLevel = 'debug' | 'info' | 'warn' | 'error';

interface LogContext {
  [key: string]: any;
}

interface LogEntry {
  timestamp: string;
  level: LogLevel;
  message: string;
  context?: LogContext;
  error?: {
    name: string;
    message: string;
    stack?: string;
  };
}

// Log level priority (higher = more severe)
const LOG_LEVELS: Record<LogLevel, number> = {
  debug: 0,
  info: 1,
  warn: 2,
  error: 3,
};

// Get configured log level from environment
function getConfiguredLevel(): LogLevel {
  const level = process.env.LOG_LEVEL?.toLowerCase() as LogLevel;
  if (level && LOG_LEVELS[level] !== undefined) {
    return level;
  }
  // Default to 'info' in production, 'debug' in development
  return process.env.NODE_ENV === 'production' ? 'info' : 'debug';
}

// Check if a log level should be output
function shouldLog(level: LogLevel): boolean {
  const configuredLevel = getConfiguredLevel();
  return LOG_LEVELS[level] >= LOG_LEVELS[configuredLevel];
}

// Format log entry for output
function formatLogEntry(entry: LogEntry): string {
  if (process.env.NODE_ENV === 'production') {
    // JSON format for production (easier to parse/analyze)
    return JSON.stringify(entry);
  }

  // Human-readable format for development
  const { timestamp, level, message, context, error } = entry;
  const levelColors: Record<LogLevel, string> = {
    debug: '\x1b[36m', // Cyan
    info: '\x1b[32m',  // Green
    warn: '\x1b[33m',  // Yellow
    error: '\x1b[31m', // Red
  };
  const reset = '\x1b[0m';
  const color = levelColors[level];

  let output = `${timestamp} ${color}[${level.toUpperCase()}]${reset} ${message}`;

  if (context && Object.keys(context).length > 0) {
    output += ` ${JSON.stringify(context)}`;
  }

  if (error) {
    output += `\n  Error: ${error.name}: ${error.message}`;
    if (error.stack && process.env.NODE_ENV !== 'production') {
      output += `\n  Stack: ${error.stack}`;
    }
  }

  return output;
}

// Create a log entry and output it
function log(level: LogLevel, message: string, context?: LogContext, error?: Error): void {
  if (!shouldLog(level)) {
    return;
  }

  const entry: LogEntry = {
    timestamp: new Date().toISOString(),
    level,
    message,
  };

  if (context && Object.keys(context).length > 0) {
    entry.context = context;
  }

  if (error) {
    entry.error = {
      name: error.name,
      message: error.message,
      stack: process.env.NODE_ENV !== 'production' ? error.stack : undefined,
    };
  }

  const formatted = formatLogEntry(entry);

  // Use appropriate console method
  switch (level) {
    case 'error':
      console.error(formatted);
      break;
    case 'warn':
      console.warn(formatted);
      break;
    default:
      console.log(formatted);
  }
}

/**
 * Logger instance with methods for each log level
 */
export const logger = {
  /**
   * Debug level - detailed information for debugging
   * Only shown when LOG_LEVEL=debug
   */
  debug(message: string, context?: LogContext): void {
    log('debug', message, context);
  },

  /**
   * Info level - general operational information
   * Shown by default
   */
  info(message: string, context?: LogContext): void {
    log('info', message, context);
  },

  /**
   * Warn level - warning conditions that should be addressed
   */
  warn(message: string, context?: LogContext): void {
    log('warn', message, context);
  },

  /**
   * Error level - error conditions
   */
  error(message: string, error?: Error | unknown, context?: LogContext): void {
    const err = error instanceof Error ? error : undefined;
    const ctx = error instanceof Error ? context : (error as LogContext);
    log('error', message, ctx, err);
  },

  /**
   * Log an HTTP request (for middleware)
   */
  request(method: string, path: string, statusCode: number, durationMs: number, context?: LogContext): void {
    const level: LogLevel = statusCode >= 500 ? 'error' : statusCode >= 400 ? 'warn' : 'info';
    log(level, `${method} ${path} ${statusCode} ${durationMs}ms`, context);
  },

  /**
   * Create a child logger with preset context
   */
  child(defaultContext: LogContext) {
    return {
      debug: (message: string, context?: LogContext) =>
        log('debug', message, { ...defaultContext, ...context }),
      info: (message: string, context?: LogContext) =>
        log('info', message, { ...defaultContext, ...context }),
      warn: (message: string, context?: LogContext) =>
        log('warn', message, { ...defaultContext, ...context }),
      error: (message: string, error?: Error | unknown, context?: LogContext) => {
        const err = error instanceof Error ? error : undefined;
        const ctx = error instanceof Error ? context : (error as LogContext);
        log('error', message, { ...defaultContext, ...ctx }, err);
      },
    };
  },
};

// Export default for convenience
export default logger;
