import * as z from "zod";
import { createTRPCRouter, publicProcedure, protectedProcedure } from "@/backend/trpc/create-context";
import { getUserData, createUser, updateUser, updatePassword } from "@/backend/db/users";
import { supabaseAdmin } from "@/backend/db/connection";
import { checkRateLimit } from "@/backend/middleware/rate-limit";
import { generateToken } from "@/backend/utils/jwt";
import { logger } from "@/backend/utils/logger";

// Helper function to validate subscription expiry
function checkSubscriptionExpiry(user: any): any {
  if (user.isPremium && user.subscriptionEndDate) {
    const expiryDate = new Date(user.subscriptionEndDate);
    const now = new Date();

    if (now > expiryDate) {
      // Subscription expired, return user with isPremium set to false
      return {
        ...user,
        isPremium: false,
        subscriptionEndDate: undefined,
      };
    }
  }
  return user;
}

export const authRouter = createTRPCRouter({
  login: publicProcedure
    .input(z.object({
      email: z.string().email(),
      password: z.string().min(6)
    }))
    .mutation(async ({ input }) => {
      if (!(await checkRateLimit(input.email, 'login'))) {
        throw new Error('Too many login attempts. Please try again later.');
      }

      // Use Supabase Auth for password verification
      const { data: authData, error: authError } = await supabaseAdmin.auth.signInWithPassword({
        email: input.email,
        password: input.password,
      });

      if (authError || !authData.user) {
        logger.warn('Login failed', { email: input.email, error: authError?.message });
        throw new Error("Invalid credentials");
      }

      // Get user profile from database
      const user = await getUserData(authData.user.id);
      if (!user) {
        logger.error('User profile not found after successful auth', { userId: authData.user.id });
        throw new Error("User profile not found. Please contact support.");
      }

      // Check subscription expiry and update if necessary
      const validatedUser = checkSubscriptionExpiry(user);
      if (validatedUser.isPremium !== user.isPremium) {
        // Subscription expired, update database
        await updateUser(user.id, { isPremium: false, subscriptionEndDate: undefined });
        logger.info('Subscription expired for user', { userId: user.id });
      }

      await updateUser(user.id, { lastLoginAt: new Date().toISOString() });

      const token = generateToken({
        userId: validatedUser.id,
        email: validatedUser.email,
        isAdmin: validatedUser.isAdmin,
      });

      logger.info('User logged in successfully', { userId: validatedUser.id });

      return {
        userId: validatedUser.id,
        email: validatedUser.email,
        name: validatedUser.name,
        isPremium: validatedUser.isPremium,
        storiesGenerated: validatedUser.storiesGenerated,
        isAdmin: validatedUser.isAdmin,
        subscriptionEndDate: validatedUser.subscriptionEndDate,
        token,
      };
    }),

  register: publicProcedure
    .input(z.object({ 
      email: z.string().email(),
      password: z.string().min(6),
      name: z.string().min(2)
    }))
    .mutation(async ({ input }) => {
      if (!(await checkRateLimit(input.email, 'register'))) {
        throw new Error('Too many registration attempts. Please try again later.');
      }
      
      const existingUser = await getUserData(input.email);
      if (existingUser) {
        throw new Error("User already exists");
      }
      
      const user = await createUser({
        email: input.email,
        password: input.password,
        name: input.name,
      });
      
      const token = generateToken({
        userId: user.id,
        email: user.email,
        isAdmin: user.isAdmin,
      });
      
      return {
        userId: user.id,
        email: user.email,
        name: user.name,
        isPremium: user.isPremium,
        storiesGenerated: user.storiesGenerated,
        isAdmin: user.isAdmin,
        subscriptionEndDate: user.subscriptionEndDate,
        token,
      };
    }),

  updateProfile: protectedProcedure
    .input(z.object({
      name: z.string().min(2).optional(),
      email: z.string().email().optional(),
    }))
    .mutation(async ({ ctx, input }) => {
      if (input.email) {
        const existingUser = await getUserData(input.email);
        if (existingUser && existingUser.id !== ctx.userId) {
          throw new Error("Email already in use");
        }
      }
      await updateUser(ctx.userId, input);
      const user = await getUserData(ctx.userId);
      if (!user) throw new Error("User not found");
      return {
        userId: user.id,
        email: user.email,
        name: user.name,
        isPremium: user.isPremium,
        storiesGenerated: user.storiesGenerated,
        isAdmin: user.isAdmin,
        subscriptionEndDate: user.subscriptionEndDate,
      };
    }),

  changePassword: protectedProcedure
    .input(z.object({
      currentPassword: z.string(),
      newPassword: z.string().min(6),
    }))
    .mutation(async ({ ctx, input }) => {
      const user = await getUserData(ctx.userId);
      if (!user) throw new Error("User not found");

      // Verify current password using Supabase Auth
      const { error: verifyError } = await supabaseAdmin.auth.signInWithPassword({
        email: user.email,
        password: input.currentPassword,
      });

      if (verifyError) {
        logger.warn('Password change failed - incorrect current password', { userId: ctx.userId });
        throw new Error("Current password is incorrect");
      }

      await updatePassword(ctx.userId, input.newPassword);
      logger.info('Password changed successfully', { userId: ctx.userId });
      return { success: true };
    }),

  me: protectedProcedure.query(async ({ ctx }) => {
    const user = await getUserData(ctx.userId);
    if (!user) {
      throw new Error("User not found");
    }
    return {
      userId: user.id,
      email: user.email,
      name: user.name,
      isPremium: user.isPremium,
      storiesGenerated: user.storiesGenerated,
      isAdmin: user.isAdmin,
      subscriptionEndDate: user.subscriptionEndDate,
    };
  }),

  upgrade: protectedProcedure.mutation(async ({ ctx }) => {
    const subscriptionEndDate = new Date();
    subscriptionEndDate.setMonth(subscriptionEndDate.getMonth() + 1);
    await updateUser(ctx.userId, { 
      isPremium: true,
      subscriptionEndDate: subscriptionEndDate.toISOString(),
    });
    return { success: true };
  }),

  cancelSubscription: protectedProcedure.mutation(async ({ ctx }) => {
    await updateUser(ctx.userId, { 
      isPremium: false,
      subscriptionEndDate: undefined,
    });
    return { success: true };
  }),
});
