import { supabaseAdmin } from './connection';
import { logger } from '../utils/logger';

type PageContent = {
  id: string;
  pageId: string;
  sectionId: string;
  type: 'text' | 'heading' | 'html' | 'image' | 'button';
  content: string;
  order: number;
  styles?: Record<string, any>;
  metadata?: Record<string, any>;
  updatedAt: string;
};

type Page = {
  id: string;
  slug: string;
  title: string;
  isPublished: boolean;
  showInMenu: boolean;
  menuOrder: number;
  showHeader: boolean;
  showFooter: boolean;
  createdAt: string;
  updatedAt: string;
  metadata?: Record<string, any>;
};

type ThemeSettings = {
  id: string;
  primaryColor: string;
  secondaryColor: string;
  accentColor: string;
  backgroundColor: string;
  textColor: string;
  headerBackgroundColor: string;
  headerTextColor: string;
  footerBackgroundColor: string;
  footerTextColor: string;
  buttonBackgroundColor: string;
  buttonTextColor: string;
  borderRadius: number;
  fontSize: number;
  fontFamily: string;
  headerHeight: number;
  footerHeight: number;
  customCSS?: string;
  updatedAt: string;
};

type HeaderFooterContent = {
  id: string;
  type: 'header' | 'footer';
  content: string;
  logo?: string;
  links?: { label: string; url: string; icon?: string }[];
  showSearch: boolean;
  showUserMenu: boolean;
  customHTML?: string;
  updatedAt: string;
};

// Helper to map database row to Page type
function mapRowToPage(row: any): Page {
  return {
    id: row.id,
    slug: row.slug,
    title: row.title,
    isPublished: row.is_published,
    showInMenu: row.show_in_menu,
    menuOrder: row.menu_order,
    showHeader: row.show_header,
    showFooter: row.show_footer,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
    metadata: row.metadata || undefined,
  };
}

// Helper to map database row to PageContent type
function mapRowToPageContent(row: any): PageContent {
  return {
    id: row.id,
    pageId: row.page_id,
    sectionId: row.section_id,
    type: row.type,
    content: row.content,
    order: row.order_num,
    styles: row.styles || undefined,
    metadata: row.metadata || undefined,
    updatedAt: row.updated_at,
  };
}

// Helper to map database row to ThemeSettings type
function mapRowToThemeSettings(row: any): ThemeSettings {
  return {
    id: row.id,
    primaryColor: row.primary_color,
    secondaryColor: row.secondary_color,
    accentColor: row.accent_color,
    backgroundColor: row.background_color,
    textColor: row.text_color,
    headerBackgroundColor: row.header_background_color,
    headerTextColor: row.header_text_color,
    footerBackgroundColor: row.footer_background_color,
    footerTextColor: row.footer_text_color,
    buttonBackgroundColor: row.button_background_color,
    buttonTextColor: row.button_text_color,
    borderRadius: row.border_radius,
    fontSize: row.font_size,
    fontFamily: row.font_family,
    headerHeight: row.header_height,
    footerHeight: row.footer_height,
    customCSS: row.custom_css || undefined,
    updatedAt: row.updated_at,
  };
}

// Helper to map database row to HeaderFooterContent type
function mapRowToHeaderFooterContent(row: any): HeaderFooterContent {
  return {
    id: row.id,
    type: row.type,
    content: row.content,
    logo: row.logo || undefined,
    links: row.links || undefined,
    showSearch: row.show_search,
    showUserMenu: row.show_user_menu,
    customHTML: row.custom_html || undefined,
    updatedAt: row.updated_at,
  };
}

// ==================== PAGE FUNCTIONS ====================

export async function getAllPages(): Promise<Page[]> {
  const { data, error } = await supabaseAdmin
    .from('pages')
    .select('*')
    .order('menu_order', { ascending: true });

  if (error) {
    logger.error('Failed to get all pages', error);
    throw new Error(error.message);
  }

  return (data || []).map(mapRowToPage);
}

export async function getPage(slugOrId: string): Promise<Page | undefined> {
  // First try to find by slug
  let { data, error } = await supabaseAdmin
    .from('pages')
    .select('*')
    .eq('slug', slugOrId)
    .maybeSingle();

  // If not found by slug, try by id
  if (!data && !error) {
    const result = await supabaseAdmin
      .from('pages')
      .select('*')
      .eq('id', slugOrId)
      .maybeSingle();
    data = result.data;
    error = result.error;
  }

  if (error) {
    logger.error('Failed to get page', error, { slugOrId });
    return undefined;
  }

  return data ? mapRowToPage(data) : undefined;
}

export async function createPage(input: Omit<Page, 'id' | 'createdAt' | 'updatedAt'>): Promise<Page> {
  const { data, error } = await supabaseAdmin
    .from('pages')
    .insert({
      slug: input.slug,
      title: input.title,
      is_published: input.isPublished,
      show_in_menu: input.showInMenu,
      menu_order: input.menuOrder,
      show_header: input.showHeader,
      show_footer: input.showFooter,
      metadata: input.metadata || null,
    })
    .select()
    .single();

  if (error || !data) {
    logger.error('Failed to create page', error);
    throw new Error(error?.message || 'Failed to create page');
  }

  logger.info('Page created', { pageId: data.id, slug: input.slug });
  return mapRowToPage(data);
}

export async function updatePage(pageId: string, updates: Partial<Page>): Promise<void> {
  const dbUpdates: any = {
    updated_at: new Date().toISOString(),
  };

  if (updates.slug !== undefined) dbUpdates.slug = updates.slug;
  if (updates.title !== undefined) dbUpdates.title = updates.title;
  if (updates.isPublished !== undefined) dbUpdates.is_published = updates.isPublished;
  if (updates.showInMenu !== undefined) dbUpdates.show_in_menu = updates.showInMenu;
  if (updates.menuOrder !== undefined) dbUpdates.menu_order = updates.menuOrder;
  if (updates.showHeader !== undefined) dbUpdates.show_header = updates.showHeader;
  if (updates.showFooter !== undefined) dbUpdates.show_footer = updates.showFooter;
  if (updates.metadata !== undefined) dbUpdates.metadata = updates.metadata;

  const { error } = await supabaseAdmin
    .from('pages')
    .update(dbUpdates)
    .eq('id', pageId);

  if (error) {
    logger.error('Failed to update page', error, { pageId });
    throw new Error(error.message);
  }

  logger.info('Page updated', { pageId });
}

export async function deletePage(pageId: string): Promise<void> {
  const { error } = await supabaseAdmin
    .from('pages')
    .delete()
    .eq('id', pageId);

  if (error) {
    logger.error('Failed to delete page', error, { pageId });
    throw new Error(error.message);
  }

  logger.info('Page deleted', { pageId });
}

// ==================== PAGE CONTENT FUNCTIONS ====================

export async function getPageContents(pageId: string): Promise<PageContent[]> {
  const { data, error } = await supabaseAdmin
    .from('page_contents')
    .select('*')
    .eq('page_id', pageId)
    .order('order_num', { ascending: true });

  if (error) {
    logger.error('Failed to get page contents', error, { pageId });
    throw new Error(error.message);
  }

  return (data || []).map(mapRowToPageContent);
}

export async function getPageContentsBySection(pageId: string, sectionId: string): Promise<PageContent[]> {
  const { data, error } = await supabaseAdmin
    .from('page_contents')
    .select('*')
    .eq('page_id', pageId)
    .eq('section_id', sectionId)
    .order('order_num', { ascending: true });

  if (error) {
    logger.error('Failed to get page contents by section', error, { pageId, sectionId });
    throw new Error(error.message);
  }

  return (data || []).map(mapRowToPageContent);
}

export async function createPageContent(input: Omit<PageContent, 'id' | 'updatedAt'>): Promise<PageContent> {
  const { data, error } = await supabaseAdmin
    .from('page_contents')
    .insert({
      page_id: input.pageId,
      section_id: input.sectionId,
      type: input.type,
      content: input.content,
      order_num: input.order,
      styles: input.styles || null,
      metadata: input.metadata || null,
    })
    .select()
    .single();

  if (error || !data) {
    logger.error('Failed to create page content', error);
    throw new Error(error?.message || 'Failed to create page content');
  }

  logger.info('Page content created', { contentId: data.id, pageId: input.pageId });
  return mapRowToPageContent(data);
}

export async function updatePageContent(contentId: string, updates: Partial<PageContent>): Promise<void> {
  const dbUpdates: any = {
    updated_at: new Date().toISOString(),
  };

  if (updates.sectionId !== undefined) dbUpdates.section_id = updates.sectionId;
  if (updates.type !== undefined) dbUpdates.type = updates.type;
  if (updates.content !== undefined) dbUpdates.content = updates.content;
  if (updates.order !== undefined) dbUpdates.order_num = updates.order;
  if (updates.styles !== undefined) dbUpdates.styles = updates.styles;
  if (updates.metadata !== undefined) dbUpdates.metadata = updates.metadata;

  const { error } = await supabaseAdmin
    .from('page_contents')
    .update(dbUpdates)
    .eq('id', contentId);

  if (error) {
    logger.error('Failed to update page content', error, { contentId });
    throw new Error(error.message);
  }

  logger.info('Page content updated', { contentId });
}

export async function deletePageContent(contentId: string): Promise<void> {
  const { error } = await supabaseAdmin
    .from('page_contents')
    .delete()
    .eq('id', contentId);

  if (error) {
    logger.error('Failed to delete page content', error, { contentId });
    throw new Error(error.message);
  }

  logger.info('Page content deleted', { contentId });
}

export async function reorderPageContents(pageId: string, sectionId: string, contentIds: string[]): Promise<void> {
  // Update order for each content item
  const updates = contentIds.map((id, index) =>
    supabaseAdmin
      .from('page_contents')
      .update({ order_num: index, updated_at: new Date().toISOString() })
      .eq('id', id)
      .eq('page_id', pageId)
      .eq('section_id', sectionId)
  );

  const results = await Promise.all(updates);
  const errors = results.filter(r => r.error);

  if (errors.length > 0) {
    logger.error('Failed to reorder some page contents', errors[0].error, { pageId, sectionId });
    throw new Error('Failed to reorder page contents');
  }

  logger.info('Page contents reordered', { pageId, sectionId, count: contentIds.length });
}

// ==================== THEME SETTINGS FUNCTIONS ====================

export async function getThemeSettings(): Promise<ThemeSettings> {
  const { data, error } = await supabaseAdmin
    .from('theme_settings')
    .select('*')
    .eq('id', 'default')
    .single();

  if (error) {
    // If no settings exist, return defaults
    if (error.code === 'PGRST116') {
      logger.info('No theme settings found, using defaults');
      return {
        id: 'default',
        primaryColor: '#6366F1',
        secondaryColor: '#8B5CF6',
        accentColor: '#EC4899',
        backgroundColor: '#FFFFFF',
        textColor: '#1F2937',
        headerBackgroundColor: '#FFFFFF',
        headerTextColor: '#1F2937',
        footerBackgroundColor: '#F9FAFB',
        footerTextColor: '#6B7280',
        buttonBackgroundColor: '#6366F1',
        buttonTextColor: '#FFFFFF',
        borderRadius: 12,
        fontSize: 16,
        fontFamily: 'System',
        headerHeight: 60,
        footerHeight: 80,
        updatedAt: new Date().toISOString(),
      };
    }
    logger.error('Failed to get theme settings', error);
    throw new Error(error.message);
  }

  return mapRowToThemeSettings(data);
}

export async function updateThemeSettings(updates: Partial<ThemeSettings>): Promise<ThemeSettings> {
  const dbUpdates: any = {
    updated_at: new Date().toISOString(),
  };

  if (updates.primaryColor !== undefined) dbUpdates.primary_color = updates.primaryColor;
  if (updates.secondaryColor !== undefined) dbUpdates.secondary_color = updates.secondaryColor;
  if (updates.accentColor !== undefined) dbUpdates.accent_color = updates.accentColor;
  if (updates.backgroundColor !== undefined) dbUpdates.background_color = updates.backgroundColor;
  if (updates.textColor !== undefined) dbUpdates.text_color = updates.textColor;
  if (updates.headerBackgroundColor !== undefined) dbUpdates.header_background_color = updates.headerBackgroundColor;
  if (updates.headerTextColor !== undefined) dbUpdates.header_text_color = updates.headerTextColor;
  if (updates.footerBackgroundColor !== undefined) dbUpdates.footer_background_color = updates.footerBackgroundColor;
  if (updates.footerTextColor !== undefined) dbUpdates.footer_text_color = updates.footerTextColor;
  if (updates.buttonBackgroundColor !== undefined) dbUpdates.button_background_color = updates.buttonBackgroundColor;
  if (updates.buttonTextColor !== undefined) dbUpdates.button_text_color = updates.buttonTextColor;
  if (updates.borderRadius !== undefined) dbUpdates.border_radius = updates.borderRadius;
  if (updates.fontSize !== undefined) dbUpdates.font_size = updates.fontSize;
  if (updates.fontFamily !== undefined) dbUpdates.font_family = updates.fontFamily;
  if (updates.headerHeight !== undefined) dbUpdates.header_height = updates.headerHeight;
  if (updates.footerHeight !== undefined) dbUpdates.footer_height = updates.footerHeight;
  if (updates.customCSS !== undefined) dbUpdates.custom_css = updates.customCSS;

  const { data, error } = await supabaseAdmin
    .from('theme_settings')
    .upsert({ id: 'default', ...dbUpdates })
    .select()
    .single();

  if (error || !data) {
    logger.error('Failed to update theme settings', error);
    throw new Error(error?.message || 'Failed to update theme settings');
  }

  logger.info('Theme settings updated');
  return mapRowToThemeSettings(data);
}

// ==================== HEADER/FOOTER FUNCTIONS ====================

export async function getHeaderFooterContent(type: 'header' | 'footer'): Promise<HeaderFooterContent | undefined> {
  const { data, error } = await supabaseAdmin
    .from('header_footer_contents')
    .select('*')
    .eq('id', type)
    .maybeSingle();

  if (error) {
    logger.error('Failed to get header/footer content', error, { type });
    return undefined;
  }

  return data ? mapRowToHeaderFooterContent(data) : undefined;
}

export async function updateHeaderFooterContent(type: 'header' | 'footer', updates: Partial<HeaderFooterContent>): Promise<void> {
  const dbUpdates: any = {
    id: type,
    type: type,
    updated_at: new Date().toISOString(),
  };

  if (updates.content !== undefined) dbUpdates.content = updates.content;
  if (updates.logo !== undefined) dbUpdates.logo = updates.logo;
  if (updates.links !== undefined) dbUpdates.links = updates.links;
  if (updates.showSearch !== undefined) dbUpdates.show_search = updates.showSearch;
  if (updates.showUserMenu !== undefined) dbUpdates.show_user_menu = updates.showUserMenu;
  if (updates.customHTML !== undefined) dbUpdates.custom_html = updates.customHTML;

  const { error } = await supabaseAdmin
    .from('header_footer_contents')
    .upsert(dbUpdates);

  if (error) {
    logger.error('Failed to update header/footer content', error, { type });
    throw new Error(error.message);
  }

  logger.info('Header/footer content updated', { type });
}

export async function getAllHeaderFooterContents(): Promise<HeaderFooterContent[]> {
  const { data, error } = await supabaseAdmin
    .from('header_footer_contents')
    .select('*');

  if (error) {
    logger.error('Failed to get all header/footer contents', error);
    throw new Error(error.message);
  }

  return (data || []).map(mapRowToHeaderFooterContent);
}

// ==================== INITIALIZATION ====================

export async function initializeDefaultContent() {
  try {
    // Check if default pages exist
    const existingPages = await getAllPages();
    if (existingPages.length === 0) {
      logger.info('Creating default pages...');

      // Create default pages
      const defaultPages = [
        { slug: 'home', title: 'Home', isPublished: true, showInMenu: true, menuOrder: 0, showHeader: true, showFooter: true },
        { slug: 'about', title: 'About Us', isPublished: true, showInMenu: true, menuOrder: 1, showHeader: true, showFooter: true },
        { slug: 'pricing', title: 'Pricing', isPublished: true, showInMenu: true, menuOrder: 2, showHeader: true, showFooter: true },
        { slug: 'faq', title: 'FAQ', isPublished: true, showInMenu: true, menuOrder: 3, showHeader: true, showFooter: true },
        { slug: 'contact', title: 'Contact', isPublished: true, showInMenu: true, menuOrder: 4, showHeader: true, showFooter: true },
        { slug: 'privacy', title: 'Privacy Policy', isPublished: true, showInMenu: false, menuOrder: 99, showHeader: true, showFooter: true },
        { slug: 'terms', title: 'Terms of Service', isPublished: true, showInMenu: false, menuOrder: 100, showHeader: true, showFooter: true },
      ];

      for (const page of defaultPages) {
        try {
          await createPage(page);
        } catch (err: any) {
          // Ignore duplicate key errors (page already exists)
          if (!err.message?.includes('duplicate')) {
            logger.warn('Failed to create default page', { slug: page.slug, error: err.message });
          }
        }
      }

      logger.info('Default pages created');
    }

    // Check if header exists
    const existingHeader = await getHeaderFooterContent('header');
    if (!existingHeader) {
      logger.info('Header content will be created when first accessed via SQL defaults');
    }

    // Check if footer exists
    const existingFooter = await getHeaderFooterContent('footer');
    if (!existingFooter) {
      logger.info('Footer content will be created when first accessed via SQL defaults');
    }

    logger.info('Content initialization complete');
  } catch (error: any) {
    logger.error('Failed to initialize default content', error);
  }
}
