import { useState, useEffect } from 'react';
import { useWindowDimensions, Platform } from 'react-native';

/**
 * Breakpoint definitions for responsive design
 */
export const BREAKPOINTS = {
  mobile: 0,
  tablet: 768,
  desktop: 1024,
  wide: 1440,
} as const;

export type Breakpoint = keyof typeof BREAKPOINTS;

/**
 * Hook to get the current breakpoint based on window width
 */
export function useBreakpoint(): Breakpoint {
  const { width } = useWindowDimensions();

  if (width >= BREAKPOINTS.wide) return 'wide';
  if (width >= BREAKPOINTS.desktop) return 'desktop';
  if (width >= BREAKPOINTS.tablet) return 'tablet';
  return 'mobile';
}

/**
 * Hook to check if the current device is desktop or larger
 */
export function useIsDesktop(): boolean {
  const breakpoint = useBreakpoint();
  return breakpoint === 'desktop' || breakpoint === 'wide';
}

/**
 * Hook to check if the current device is tablet or larger
 */
export function useIsTablet(): boolean {
  const breakpoint = useBreakpoint();
  return breakpoint === 'tablet' || breakpoint === 'desktop' || breakpoint === 'wide';
}

/**
 * Hook to check if running on web platform
 */
export function useIsWeb(): boolean {
  return Platform.OS === 'web';
}

/**
 * Responsive utility functions for styling
 */
export const responsive = {
  /**
   * Get responsive padding based on breakpoint
   */
  padding: (breakpoint: Breakpoint): number => {
    switch (breakpoint) {
      case 'wide': return 48;
      case 'desktop': return 40;
      case 'tablet': return 32;
      case 'mobile': return 24;
    }
  },

  /**
   * Get responsive margin based on breakpoint
   */
  margin: (breakpoint: Breakpoint): number => {
    switch (breakpoint) {
      case 'wide': return 32;
      case 'desktop': return 24;
      case 'tablet': return 20;
      case 'mobile': return 16;
    }
  },

  /**
   * Get responsive font size based on breakpoint
   */
  fontSize: (base: number, breakpoint: Breakpoint): number => {
    const multiplier = {
      wide: 1.2,
      desktop: 1.1,
      tablet: 1.05,
      mobile: 1,
    };
    return Math.round(base * multiplier[breakpoint]);
  },

  /**
   * Get responsive container max width
   */
  maxWidth: (breakpoint: Breakpoint): number | string => {
    switch (breakpoint) {
      case 'wide': return 1280;
      case 'desktop': return 1024;
      case 'tablet': return 768;
      case 'mobile': return '100%';
    }
  },

  /**
   * Get number of columns for grid layouts
   */
  columns: (breakpoint: Breakpoint): number => {
    switch (breakpoint) {
      case 'wide': return 4;
      case 'desktop': return 3;
      case 'tablet': return 2;
      case 'mobile': return 1;
    }
  },
};

/**
 * Hook to get responsive values that update on window resize
 */
export function useResponsive() {
  const breakpoint = useBreakpoint();
  const isDesktop = useIsDesktop();
  const isTablet = useIsTablet();
  const isWeb = useIsWeb();

  return {
    breakpoint,
    isDesktop,
    isTablet,
    isWeb,
    padding: responsive.padding(breakpoint),
    margin: responsive.margin(breakpoint),
    maxWidth: responsive.maxWidth(breakpoint),
    columns: responsive.columns(breakpoint),
    fontSize: (base: number) => responsive.fontSize(base, breakpoint),
  };
}

/**
 * Helper to create responsive styles based on breakpoint
 */
export function createResponsiveStyle<T>(
  mobile: T,
  tablet?: T,
  desktop?: T,
  wide?: T
): (breakpoint: Breakpoint) => T {
  return (breakpoint: Breakpoint) => {
    switch (breakpoint) {
      case 'wide': return wide ?? desktop ?? tablet ?? mobile;
      case 'desktop': return desktop ?? tablet ?? mobile;
      case 'tablet': return tablet ?? mobile;
      case 'mobile': return mobile;
    }
  };
}
