/**
 * PM2 Ecosystem Configuration
 *
 * This file configures PM2 for process management on cPanel.
 *
 * Usage:
 *   pm2 start ecosystem.config.js
 *   pm2 restart socialstoryai
 *   pm2 logs socialstoryai
 *   pm2 monit
 *
 * Note: cPanel's built-in Node.js manager may not support PM2.
 * Check if PM2 is available: `which pm2` or install globally if allowed.
 */

module.exports = {
  apps: [
    {
      // Application name (shown in pm2 list)
      name: 'socialstoryai',

      // Entry point
      script: 'backend/hono.js',

      // Working directory
      cwd: __dirname,

      // Node.js arguments
      node_args: '--max-old-space-size=512',

      // Environment variables (loaded from .env file)
      env: {
        NODE_ENV: 'production',
        PORT: 3000,
      },

      // Instance configuration
      instances: 1, // Use 1 for shared hosting (limited resources)
      exec_mode: 'fork', // 'cluster' mode requires more memory

      // Auto-restart configuration
      autorestart: true,
      watch: false, // Don't watch files in production
      max_memory_restart: '500M', // Restart if memory exceeds 500MB

      // Crash recovery
      min_uptime: '10s', // Consider app started if running 10s+
      max_restarts: 10, // Max restarts within min_uptime window
      restart_delay: 4000, // Wait 4s before restarting

      // Logging
      error_file: 'logs/error.log',
      out_file: 'logs/output.log',
      log_file: 'logs/combined.log',
      time: true, // Add timestamps to logs
      merge_logs: true, // Merge logs from all instances

      // Log rotation (keeps logs manageable)
      log_date_format: 'YYYY-MM-DD HH:mm:ss Z',

      // Graceful shutdown
      kill_timeout: 5000, // Wait 5s for graceful shutdown
      listen_timeout: 3000, // Wait 3s for app to listen on port

      // Health check (if PM2 Plus is available)
      // Uncomment if using PM2 Plus monitoring
      // exp_backoff_restart_delay: 100,
    },
  ],

  // Deployment configuration (for pm2 deploy)
  // Uncomment and configure if using pm2 deploy feature
  /*
  deploy: {
    production: {
      user: 'your-cpanel-username',
      host: 'your-server.com',
      ref: 'origin/main',
      repo: 'git@github.com:username/socialstoryai.git',
      path: '/home/username/socialstoryai',
      'pre-deploy': 'git fetch --all',
      'post-deploy': 'npm install && pm2 reload ecosystem.config.js --env production',
      env: {
        NODE_ENV: 'production',
      },
    },
  },
  */
};
